
#ifndef V3D_BASICS_MATRIXTRANSF_TPL_H_INCLUDED
#define V3D_BASICS_MATRIXTRANSF_TPL_H_INCLUDED

#include "Vector.h"
#include "Maths.h"


namespace V3D {

template< typename REAL > class MatrixProjTpl;


void TridiagonalNoTemplate(Vector3f avcCols[3], float afDiag[3], float afSubDiag[3]);
bool QLAlgorithmNoTemplate(Vector3f avcCols[3], float afDiag[3], float afSubDiag[3]);


template< typename REAL >
class MatrixTransfTpl    // Matrice de transformation affine
                        // (4*4 dont la derniere ligne est 0,0,0,1)
                        // La matrice est representee comme une 4*3
{
	friend class MatrixProjTpl<REAL>;

public:
	static const MatrixTransfTpl<REAL> k_mtxIdentity;
	static const MatrixTransfTpl<REAL> k_mtxZero;

public:
	// ATTENTION : Classe concrete. Pas de virtuel dans la classe. Ne doit pas etre derivee.
	MatrixTransfTpl() {}
	MatrixTransfTpl(const Vector3f& vcCol1, const Vector3f& vcCol2, const Vector3f& vcCol3, const Vector3Tpl<REAL>& vcTransl = Vector3Tpl<REAL>(0,0,0));
	//explicit MatrixTransfTpl(int iii) {}

	bool Invert();
	void InvertMvt();	// Mouvement = Rotation + translation uniquement

	void TransposeSquarePart(); // Transposition de la partie 3*3 de gauche (tout sauf translation)

	// Operateur de multiplication droite de matrices de transformation
	// (On ne veut pas de la multiplication simple (operateur *) pour eviter la construction d'objets temporaires inutiles)
	MatrixTransfTpl<REAL>& operator *= (const MatrixTransfTpl<REAL>& mat);
	MatrixTransfTpl<REAL>& LeftMult(const MatrixTransfTpl<REAL>& mat);

	inline float GetTrace() const;

	const Vector3Tpl<REAL>& GetTranslation() const;
	void GetTranslation(Vector3Tpl<REAL>& vc) const;

	void SetTranslation(REAL tx, REAL ty, REAL tz);
	void SetTranslation(const Vector3Tpl<REAL>& vec);

//	const Vector3f& GetColumn(int32 nIndex) const;
	const Vector3f& GetColumnX() const;
	const Vector3f& GetColumnY() const;
	const Vector3f& GetColumnZ() const;


	void GetColumns( Vector3f& v1, Vector3f& v2, Vector3f& v3) const;
	void GetColumns( Vector4f& v1, Vector4f& v2, Vector4f& v3, Vector4Tpl<REAL>& v4) const;
	void SetColumns( const Vector3f& v1, const Vector3f& v2, const Vector3f& v3);

	void SetColumn(int32 nIndex, const Vector3f& vec);
	void SetColumn(int32 nIndex, float cx, float cy, float cz);

	void RoundToZero();

	void LoadIdentity();
	void MakeDiagonal( float dx, float dy, float dz);
	void MakeZero();

	MatrixTransfTpl<REAL> GetNormalsTransform() const;

	bool IsEqual( const MatrixTransfTpl& mtx, float fEpsilon = 1E-6f ) const;

	float GetDeterminant() const;

	void GetOGLMatrix( float pafMatEntries[16] ) const;

	void ComputeSymmetricMatrixEigenVectors( Vector3f& vcEig1, Vector3f& vcEig2, Vector3f& vcEig3) const;


	template <typename REALOTHER>
		inline Vector3Tpl<REALOTHER> operator * (const Vector3Tpl<REALOTHER>& vec) const
		{
			Vector3Tpl<REALOTHER> res = vec;
			Transform( res );
			return res;
		}


	template <typename REALOTHER>
		void Transform( Vector3Tpl<REALOTHER>& vc) const
		{
			vc.Set( m_avcCols[0].x * vc.x + m_avcCols[1].x * vc.y + m_avcCols[2].x * vc.z + m_vcTransl.x,
			        m_avcCols[0].y * vc.x + m_avcCols[1].y * vc.y + m_avcCols[2].y * vc.z + m_vcTransl.y,
					m_avcCols[0].z * vc.x + m_avcCols[1].z * vc.y + m_avcCols[2].z * vc.z + m_vcTransl.z );
		}

	template <typename REALOTHER>
		void TransformDirection( Vector3Tpl<REALOTHER>& vc) const
		{
			vc.Set( m_avcCols[0].x * vc.x + m_avcCols[1].x * vc.y + m_avcCols[2].x * vc.z,
			        m_avcCols[0].y * vc.x + m_avcCols[1].y * vc.y + m_avcCols[2].y * vc.z,
					m_avcCols[0].z * vc.x + m_avcCols[1].z * vc.y + m_avcCols[2].z * vc.z );
		}


	bool IsSolidTransform() const;
	bool HasShear() const;

	void ExtractRotScaleShear( MatrixTransfTpl<REAL>& mtxRotPos, Vector3f& vcScale, float& fShearXY, float& fShearXZ, float& fShearYZ) const;

	static const MatrixTransfTpl& GetIdentityInstance() { return k_mtxIdentity; }
	static const MatrixTransfTpl& GetZeroMatrixInstance() { return k_mtxZero; }

private:
	// Partie rotation/scale en float car moins besoin de precision par rapport a la translation
	Vector3f         m_avcCols[3];
	Vector3Tpl<REAL> m_vcTransl;

private:
	// Definis dans MatrixTransf.cpp
	friend void TridiagonalNoTemplate(Vector3f avcCols[3], float afDiag[3], float afSubDiag[3]);
	friend bool QLAlgorithmNoTemplate(Vector3f avcCols[3], float afDiag[3], float afSubDiag[3]);


	void Tridiagonal(float afDiag[3], float afSubDiag[3])
	{ TridiagonalNoTemplate( m_avcCols, afDiag, afSubDiag); }

	bool QLAlgorithm(float afDiag[3], float afSubDiag[3])
	{ return QLAlgorithmNoTemplate( m_avcCols, afDiag, afSubDiag); }
};


//////////////////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////////////////



template <typename REAL>
MatrixTransfTpl<REAL>::MatrixTransfTpl(const Vector3f& vcCol1, const Vector3f& vcCol2, const Vector3f& vcCol3,
                                 const Vector3Tpl<REAL>& vcTransl /* = Vector3Tpl<REAL>(0,0,0) */)
: m_vcTransl( vcTransl)
{
	m_avcCols[0] = vcCol1;
	m_avcCols[1] = vcCol2;
	m_avcCols[2] = vcCol3;
 }



template <typename REAL>
inline float MatrixTransfTpl<REAL>::GetTrace() const
{
	return m_avcCols[0][0] + m_avcCols[1][1] + m_avcCols[2][2] + 1.f;
}



template <typename REAL>
const Vector3Tpl<REAL>& MatrixTransfTpl<REAL>::GetTranslation() const
{
	return m_vcTransl;
}

template <typename REAL>
void MatrixTransfTpl<REAL>::GetTranslation(Vector3Tpl<REAL>& vc) const
{
	vc.Set( m_vcTransl );
}


template <typename REAL>
void MatrixTransfTpl<REAL>::SetTranslation(const Vector3Tpl<REAL>& vec)
{
	m_vcTransl.Set( vec );
}



template <typename REAL>
void MatrixTransfTpl<REAL>::SetTranslation(REAL tx, REAL ty, REAL tz)
{
	m_vcTransl.Set( tx, ty, tz);
}


template <typename REAL>
void MatrixTransfTpl<REAL>::MakeZero()
{
	m_avcCols[0].Set(0,0,0);
	m_avcCols[1].Set(0,0,0);
	m_avcCols[2].Set(0,0,0);
	m_vcTransl.Set(0,0,0);
}


template <typename REAL>
void MatrixTransfTpl<REAL>::MakeDiagonal( float dx, float dy, float dz)
{
	m_avcCols[0].Set( dx, 0, 0);
	m_avcCols[1].Set( 0, dy, 0);
	m_avcCols[2].Set( 0, 0, dz);
	m_vcTransl.Set(0,0,0);
}

template <typename REAL>
void MatrixTransfTpl<REAL>::LoadIdentity()
{
	MakeDiagonal( 1.f, 1.f, 1.f);
}



//template <typename REAL>
//const Vector3f& MatrixTransfTpl<REAL>::GetColumn(int32 nIndex) const
//{
//	assert( nIndex >= 0 && nIndex < 3 );
//	return m_avcCols[nIndex];
//}

template <typename REAL>
const Vector3f& MatrixTransfTpl<REAL>::GetColumnX() const
{
	return m_avcCols[0];
}

template <typename REAL>
const Vector3f& MatrixTransfTpl<REAL>::GetColumnY() const
{
	return m_avcCols[1];
}

template <typename REAL>
const Vector3f& MatrixTransfTpl<REAL>::GetColumnZ() const
{
	return m_avcCols[2];
}


template <typename REAL>
void MatrixTransfTpl<REAL>::SetColumn(int32 nIndex, const Vector3f& vec)
{
	assert( nIndex >= 0 && nIndex < 3 );
	m_avcCols[nIndex] = vec;
}

template <typename REAL>
void MatrixTransfTpl<REAL>::SetColumn(int32 nIndex, float cx, float cy, float cz)
{
	assert( nIndex >= 0 && nIndex < 3 );
	m_avcCols[nIndex].Set( cx, cy, cz );
}

template <typename REAL>
void MatrixTransfTpl<REAL>::GetColumns(Vector3f& v1, Vector3f& v2, Vector3f& v3) const
{
	v1 = m_avcCols[0];
	v2 = m_avcCols[1];
	v3 = m_avcCols[2];
}


template <typename REAL>
void MatrixTransfTpl<REAL>::GetColumns( Vector4f& v1, Vector4f& v2, Vector4f& v3, Vector4Tpl<REAL>& v4) const
{
	v1.Set( m_avcCols[0], 0.f );
	v2.Set( m_avcCols[1], 0.f );
	v3.Set( m_avcCols[2], 0.f );
	v4.Set( m_vcTransl, 1.f);
}


template <typename REAL>
void MatrixTransfTpl<REAL>::SetColumns(const Vector3f& v1, const Vector3f& v2, const Vector3f& v3)
{
	m_avcCols[0] = v1;
	m_avcCols[1] = v2;
	m_avcCols[2] = v3;
}


template <typename REAL>
float MatrixTransfTpl<REAL>::GetDeterminant() const
{
	// Le determinant d'une matrice 4*4 dont la derniere ligne est (0, 0, 0, 1) est egal
	// au determinant de la matrice 3*3
	float  m11 = m_avcCols[0].x, m12 = m_avcCols[1].x, m13 = m_avcCols[2].x;
	float  m21 = m_avcCols[0].y, m22 = m_avcCols[1].y, m23 = m_avcCols[2].y;
	float  m31 = m_avcCols[0].z, m32 = m_avcCols[1].z, m33 = m_avcCols[2].z;

	float fDet11 = m22 * m33 - m32 * m23;	// | m11  m12  m13 |
	float fDet12 = m21 * m33 - m31 * m23;	// | m21  m22  m23 |
	float fDet13 = m21 * m32 - m31 * m22;	// | m31  m32  m33 |

	return (m11 * fDet11) - (m12 * fDet12) + (m13 * fDet13);
}

template <typename REAL>
bool MatrixTransfTpl<REAL>::IsEqual( const MatrixTransfTpl<REAL>& mtx, float fEpsilon /*=  1E-6f */ ) const
{
	return (   m_avcCols[0].IsEqual( mtx.m_avcCols[0], fEpsilon )
            && m_avcCols[1].IsEqual( mtx.m_avcCols[1], fEpsilon )
            && m_avcCols[2].IsEqual( mtx.m_avcCols[2], fEpsilon )
            && m_vcTransl.IsEqual( mtx.m_vcTransl, fEpsilon ) );
}




template <typename REAL>
MatrixTransfTpl<REAL>& MatrixTransfTpl<REAL>::LeftMult(const MatrixTransfTpl<REAL>& mat)
{
	float      m11 = mat.m_avcCols[0].x, m21 = mat.m_avcCols[0].y, m31 = mat.m_avcCols[0].z;
	float      m12 = mat.m_avcCols[1].x, m22 = mat.m_avcCols[1].y, m32 = mat.m_avcCols[1].z;
	float      m13 = mat.m_avcCols[2].x, m23 = mat.m_avcCols[2].y, m33 = mat.m_avcCols[2].z;
	REAL       m14 = mat.m_vcTransl.x,   m24 = mat.m_vcTransl.y,   m34 = mat.m_vcTransl.z;

	float      n11 = m_avcCols[0].x, n21 = m_avcCols[0].y, n31 = m_avcCols[0].z;
	float      n12 = m_avcCols[1].x, n22 = m_avcCols[1].y, n32 = m_avcCols[1].z;
	float      n13 = m_avcCols[2].x, n23 = m_avcCols[2].y, n33 = m_avcCols[2].z;
	REAL       n14 = m_vcTransl.x,   n24 = m_vcTransl.y,   n34 = m_vcTransl.z;

	// | m11  m12  m13  m14 |   | n11  n12  n13  n14 |
	// | m21  m22  m23  m24 | * | n21  n22  n23  n24 |
	// | m31  m32  m33  m34 |   | n31  n32  n33  n34 |
	// |  0    0    0    1  |   |  0    0    0    1  |

	m_avcCols[0].Set( m11 * n11 + m12 * n21 + m13 * n31,
	                  m21 * n11 + m22 * n21 + m23 * n31,
	                  m31 * n11 + m32 * n21 + m33 * n31);

	m_avcCols[1].Set( m11 * n12 + m12 * n22 + m13 * n32,
	                  m21 * n12 + m22 * n22 + m23 * n32,
	                  m31 * n12 + m32 * n22 + m33 * n32);

	m_avcCols[2].Set( m11 * n13 + m12 * n23 + m13 * n33,
	                  m21 * n13 + m22 * n23 + m23 * n33,
	                  m31 * n13 + m32 * n23 + m33 * n33);

	m_vcTransl.Set( m11 * n14 + m12 * n24 + m13 * n34 + m14,
	                m21 * n14 + m22 * n24 + m23 * n34 + m24,
	                m31 * n14 + m32 * n24 + m33 * n34 + m34);
	return *this;
}



template <typename REAL>
MatrixTransfTpl<REAL>& MatrixTransfTpl<REAL>::operator *= (const MatrixTransfTpl<REAL>& mat)	// Multiplication droite
{
	float      m11 = m_avcCols[0].x, m21 = m_avcCols[0].y, m31 = m_avcCols[0].z;
	float      m12 = m_avcCols[1].x, m22 = m_avcCols[1].y, m32 = m_avcCols[1].z;
	float      m13 = m_avcCols[2].x, m23 = m_avcCols[2].y, m33 = m_avcCols[2].z;
	REAL       m14 = m_vcTransl.x,   m24 = m_vcTransl.y,   m34 = m_vcTransl.z;

	float      n11 = mat.m_avcCols[0].x, n21 = mat.m_avcCols[0].y, n31 = mat.m_avcCols[0].z;
	float      n12 = mat.m_avcCols[1].x, n22 = mat.m_avcCols[1].y, n32 = mat.m_avcCols[1].z;
	float      n13 = mat.m_avcCols[2].x, n23 = mat.m_avcCols[2].y, n33 = mat.m_avcCols[2].z;
	REAL       n14 = mat.m_vcTransl.x,   n24 = mat.m_vcTransl.y,   n34 = mat.m_vcTransl.z;

	// | m11  m12  m13  m14 |   | n11  n12  n13  n14 |
	// | m21  m22  m23  m24 | * | n21  n22  n23  n24 |
	// | m31  m32  m33  m34 |   | n31  n32  n33  n34 |
	// |  0    0    0    1  |   |  0    0    0    1  |

	m_avcCols[0].Set( m11 * n11 + m12 * n21 + m13 * n31,
	                  m21 * n11 + m22 * n21 + m23 * n31,
	                  m31 * n11 + m32 * n21 + m33 * n31);

	m_avcCols[1].Set( m11 * n12 + m12 * n22 + m13 * n32,
	                  m21 * n12 + m22 * n22 + m23 * n32,
	                  m31 * n12 + m32 * n22 + m33 * n32);

	m_avcCols[2].Set( m11 * n13 + m12 * n23 + m13 * n33,
	                  m21 * n13 + m22 * n23 + m23 * n33,
	                  m31 * n13 + m32 * n23 + m33 * n33);

	m_vcTransl.Set( m11 * n14 + m12 * n24 + m13 * n34 + m14,
	                m21 * n14 + m22 * n24 + m23 * n34 + m24,
	                m31 * n14 + m32 * n24 + m33 * n34 + m34);
	return *this;
}





template <typename REAL>
bool MatrixTransfTpl<REAL>::Invert()
{
	double det = GetDeterminant();

	if( fabs( det ) < 0.0000001)
		return false;

	// Inversion de la matrice 3x3 dans un premier temps
	double d0 = m_avcCols[0].x;
	double d1 = m_avcCols[1].x;
	double d2 = m_avcCols[2].x;

	double d3 = m_avcCols[0].y;
	double d4 = m_avcCols[1].y;
	double d5 = m_avcCols[2].y;

	double d6 = m_avcCols[0].z;
	double d7 = m_avcCols[1].z;
	double d8 = m_avcCols[2].z;

	double invdet = 1.0 / det;
	m_avcCols[0].Set( float(  ( d4 * d8 - d5 * d7) * invdet),
	                  float( -( d3 * d8 - d5 * d6) * invdet),
	                  float(  ( d3 * d7 - d6 * d4) * invdet) );

	m_avcCols[1].Set( float( -( d1 * d8 - d7 * d2) * invdet),
	                  float(  ( d0 * d8 - d6 * d2) * invdet),
	                  float( -( d0 * d7 - d6 * d1) * invdet) );

	m_avcCols[2].Set( float(  ( d1 * d5 - d4 * d2) * invdet),
	                  float( -( d0 * d5 - d3 * d2) * invdet),
	                  float(  ( d0 * d4 - d1 * d3) * invdet) );


	// Inversion au niveau de la translation
	Vector3Tpl<REAL> vcTrans = -GetTranslation();
	SetTranslation(0, 0, 0);

	vcTrans = (*this) * vcTrans;
	SetTranslation(vcTrans);
	return true;
}



template <typename REAL>
void MatrixTransfTpl<REAL>::InvertMvt()
{
	// Transposition de la matrice 3x3 que l'on suppose de rotation
	TransposeSquarePart();

	Vector3Tpl<REAL> vcTrans = -GetTranslation();
	SetTranslation(0, 0, 0);

	vcTrans = (*this) * vcTrans;
	SetTranslation(vcTrans);
}


template <typename REAL>
void MatrixTransfTpl<REAL>::TransposeSquarePart()
{
	// Transposition de la matrice 3x3
	std::swap( m_avcCols[0].y, m_avcCols[1].x);
	std::swap( m_avcCols[0].z, m_avcCols[2].x);
	std::swap( m_avcCols[1].z, m_avcCols[2].y);
}

template <typename REAL>
MatrixTransfTpl<REAL> MatrixTransfTpl<REAL>::GetNormalsTransform() const
{
	// matrice des normales = (M^-1)^T, sans translation
	MatrixTransfTpl<REAL> matResult = *this;
	matResult.SetTranslation(0.f, 0.f, 0.f);
	matResult.Invert();
	matResult.TransposeSquarePart();
	return matResult;
}




template <typename REAL>
void MatrixTransfTpl<REAL>::RoundToZero()
{
	MatrixTransfTpl<REAL> matAbs = *this;

	// Recherche de la valeur absolue maximale + construction matrice valeurs absolues
	REAL fMaxAbsVal = 0;
	for( int32 i = 0 ; i < 3; ++i)
	{
		matAbs.m_avcCols[i].x = Abs( m_avcCols[i].x );
		matAbs.m_avcCols[i].y = Abs( m_avcCols[i].y );
		matAbs.m_avcCols[i].z = Abs( m_avcCols[i].z );
		if( fMaxAbsVal < matAbs.m_avcCols[i].x ) fMaxAbsVal = matAbs.m_avcCols[i].x;
		if( fMaxAbsVal < matAbs.m_avcCols[i].y ) fMaxAbsVal = matAbs.m_avcCols[i].y;
		if( fMaxAbsVal < matAbs.m_avcCols[i].z ) fMaxAbsVal = matAbs.m_avcCols[i].z;
	}

	matAbs.m_vcTransl.x = Abs( m_vcTransl.x );
	matAbs.m_vcTransl.y = Abs( m_vcTransl.y );
	matAbs.m_vcTransl.z = Abs( m_vcTransl.z );
	if( fMaxAbsVal < matAbs.m_vcTransl.x ) fMaxAbsVal = matAbs.m_vcTransl.x;
	if( fMaxAbsVal < matAbs.m_vcTransl.y ) fMaxAbsVal = matAbs.m_vcTransl.y;
	if( fMaxAbsVal < matAbs.m_vcTransl.z ) fMaxAbsVal = matAbs.m_vcTransl.z;

	REAL fEpsilon = fMaxAbsVal * 1E-6f;

	for( int32 i = 0 ; i < 3; ++i)
	{
		if( matAbs.m_avcCols[i].x <= fEpsilon ) m_avcCols[i].x = 0;
		if( matAbs.m_avcCols[i].y <= fEpsilon ) m_avcCols[i].y = 0;
		if( matAbs.m_avcCols[i].z <= fEpsilon ) m_avcCols[i].z = 0;
	}
	if( matAbs.m_vcTransl.x <= fEpsilon ) m_vcTransl.x = 0;
	if( matAbs.m_vcTransl.y <= fEpsilon ) m_vcTransl.y = 0;
	if( matAbs.m_vcTransl.z <= fEpsilon ) m_vcTransl.z = 0;
}





template <typename REAL>
void MatrixTransfTpl<REAL>::GetOGLMatrix( float pafMatEntries[16] ) const
{
	pafMatEntries[ 0] = m_avcCols[0].x;
	pafMatEntries[ 1] = m_avcCols[0].y;
	pafMatEntries[ 2] = m_avcCols[0].z;
	pafMatEntries[ 3] = 0;

	pafMatEntries[ 4] = m_avcCols[1].x;
	pafMatEntries[ 5] = m_avcCols[1].y;
	pafMatEntries[ 6] = m_avcCols[1].z;
	pafMatEntries[ 7] = 0;

	pafMatEntries[ 8] = m_avcCols[2].x;
	pafMatEntries[ 9] = m_avcCols[2].y;
	pafMatEntries[10] = m_avcCols[2].z;
	pafMatEntries[11] = 0;

	pafMatEntries[12] = float(m_vcTransl.x);
	pafMatEntries[13] = float(m_vcTransl.y);
	pafMatEntries[14] = float(m_vcTransl.z);
	pafMatEntries[15] = 1;
}




template <typename REAL>
bool MatrixTransfTpl<REAL>::IsSolidTransform() const
{
	return (   fabs( m_avcCols[0] * m_avcCols[1] ) +
			   fabs( m_avcCols[0] * m_avcCols[2] ) +
			   fabs( m_avcCols[1] * m_avcCols[2] ) +
			   fabs( m_avcCols[0].GetSquareLength() - 1.0 ) +
			   fabs( m_avcCols[1].GetSquareLength() - 1.0 ) +
			   fabs( GetDeterminant() - 1.0 ) < 1E-5 );
}

template <typename REAL>
bool MatrixTransfTpl<REAL>::HasShear() const
{
	return (   fabs( m_avcCols[0] * m_avcCols[1] ) +
			   fabs( m_avcCols[0] * m_avcCols[2] ) +
			   fabs( m_avcCols[1] * m_avcCols[2] ) > 1E-5 );
}


template <typename REAL>
void MatrixTransfTpl<REAL>::ExtractRotScaleShear( MatrixTransfTpl<REAL>& mtxRotPos, Vector3f& vcScale, float& fShearXY, float& fShearXZ, float& fShearYZ) const
{
	const Vector3f& vcMatX = m_avcCols[0];
	const Vector3f& vcMatY = m_avcCols[1];
	const Vector3f& vcMatZ = m_avcCols[2];

	MatrixTransfTpl<REAL> mtxExtract = *this;

	// On extrait d'abord la translation (trivial)
	mtxRotPos.SetTranslation( m_vcTransl );

	// Passage de  T * R * Sh * Scale  a    R * Sh * Scale
	// mtxExtract.SetTranslation(0,0,0);  // Inutile, T est independant des autres matrices

	// On extrait la rotation R : Elle transforme l'axe des X en le vecteur mtx.X
	//                            et qui transforme le vecteur vcRotY (qui est sur le plan Z=0) en mtx.y

	// R transform (1,0,0) en vcRotX (on fait abstraction du scale en normalisant le vecteur)

	Vector3f vcRotX, vcRotY, vcRotZ;

	// Traitement des cas :
	//  - (matX et matY non colineaires, equivalent a sx * sy != 0)
	//    Note : pas equivalent a vcMatX et vcMatY !=0 !!!
	if(  (vcMatX ^ vcMatY).GetSquareLength() > 1E-12 )
	{
		vcRotX = Normalized(vcMatX);

		// mtxRot transforme la direction (1,0,0) en vcMatX et (a,b,0) en vcMatY (a la norme pres)
		// Donc mtxRot transforme (0,1,0) en (vcMatY - a*vcRotX)   (tjs a la norme pres)
		// De plus mtxRot * (1,0,0)   est orthogonale a  mtxRot * (0,1,0)
		// D'ou vcRotX * (vcMatY - a*vcRotX) == 0
		// et donc  a = (vcMatY * vcRotX) / vcRotX^2
		//            = (vcMatY * vcRotX)

		vcRotY = Normalized( vcMatY - vcRotX * (vcRotX * vcMatY) );
		vcRotZ = vcRotX ^ vcRotY;
	}
	//  - (matY et matZ non colineaires, equivalent a sy * sz != 0)
	//  -> Suppose matX liee a matY ou matZ, ca permet de definir shearXY et shearXZ = 0
	else if(  (vcMatY ^ vcMatZ).GetSquareLength() > 1E-6 )
	{
		vcRotX = Normalized(vcMatY ^ vcMatZ);
		vcRotY = Normalized(vcMatY);
		vcRotZ = vcRotX ^ vcRotY;
	}
	//  - (matX et matZ non colineaires, equivalent a sx * sz != 0)
	//  -> Suppose matY liee a matX ou matZ, ca permet de definir shearXY et shearYZ = 0
	else if(  (vcMatX ^ vcMatZ).GetSquareLength() > 1E-6 )
	{
		vcRotX = Normalized(vcMatX);
		vcRotY = Normalized(vcMatZ ^ vcMatX);
		vcRotZ = vcRotX ^ vcRotY;
	}
	// matX, matY et matZ sont colineaires : matX != 0
	else if( vcMatX.GetSquareLength() > 1E-6 )
	{
		// Cas ou un des vecteurs n'est pas nul
		//   -> R transforme (1,0,0) en ce vecteur
		vcRotX = Normalized(vcMatX);
		if( fabsf(vcRotX.y) < .99f )
			vcRotZ = Normalized( vcRotX ^ Vector3f(0,-1,0) );
		else
			vcRotZ = Normalized( vcRotX ^ Vector3f(0,0,1) );
		vcRotY = vcRotZ ^ vcRotX;
	}
	// matX, matY et matZ sont colineaires
	else if( vcMatY.GetSquareLength() > 1E-6 )
	{
		vcRotY = Normalized(vcMatY);
		if( fabsf(vcRotY.y) < .99f )
			vcRotZ = Normalized( vcRotY ^ Vector3f(0,1,0) );
		else
			vcRotZ = Normalized( vcRotY ^ Vector3f(0,0,1) );
		vcRotX = vcRotY ^ vcRotZ;
	}
	// matX, matY et matZ sont colineaires
	else if( vcMatZ.GetSquareLength() > 1E-6 )
	{
		vcRotZ = Normalized(vcMatZ);
		if( fabsf(vcRotZ.y) < .99f )
			vcRotX = Normalized( vcRotZ ^ Vector3f(0,1,0) );
		else
			vcRotX = Normalized( vcRotZ ^ Vector3f(0,0,1) );
		vcRotY = vcRotZ ^ vcRotX;
	}
	else  // scaleX = scaleY = scaleZ = 0
	{
		vcRotX.Set(1,0,0);
		vcRotY.Set(0,1,0);
		vcRotZ.Set(0,0,1);
	}


	// On connait maintenant la matrice de rotation
	mtxRotPos.SetColumns( vcRotX, vcRotY, vcRotZ );
	mtxRotPos.SetTranslation( m_vcTransl );

	// Passage de  T * R * Sh * Scale  a    Sh * Scale
	MatrixTransfTpl<REAL> mtxRotPosInv = mtxRotPos;
	mtxRotPosInv.InvertMvt();

	mtxExtract.LeftMult( mtxRotPosInv );

	//               | sx  sy*shXY sz*shXZ |
	//  mtxExtract = | 0     sy    sz*shYZ |
	//               | 0     0       sz

	Vector3f vcCol1, vcCol2, vcCol3;
	mtxExtract.GetColumns( vcCol1, vcCol2, vcCol3 );
	vcScale.Set( vcCol1.x, vcCol2.y, vcCol3.z );

	if( fabsf( vcCol2.y ) > 1E-6 )
		fShearXY = vcCol2.x / vcCol2.y;
	else
		fShearXY = 0.f;

	if( fabsf( vcCol3.z ) > 1E-6 )
	{
		fShearXZ = vcCol3.x / vcCol3.z;
		fShearYZ = vcCol3.y / vcCol3.z;
	}
	else
	{
		fShearXZ = 0.f;
		fShearYZ = 0.f;
	}
}





// Par Dave Eberly
template <typename REAL>
void MatrixTransfTpl<REAL>::ComputeSymmetricMatrixEigenVectors( Vector3f& vcEig1, Vector3f& vcEig2, Vector3f& vcEig3) const
{
	float    afEigenvalue[3];
	Vector3f akEigenvector[3];

	MatrixTransfTpl<REAL> kMatrix = *this;

	float afSubDiag[3];
	kMatrix.Tridiagonal(afEigenvalue,afSubDiag);
	kMatrix.QLAlgorithm(afEigenvalue,afSubDiag);

	kMatrix.GetColumns( vcEig1, vcEig2, vcEig3);

	// Make eigenvectors form a right--handed system
	const REAL fDet = kMatrix.GetDeterminant();
	if( fDet < 0.f )
		vcEig3 *= -1.f;
}


} // namespaces



#endif		//#ifndef V3D_BASICS_MATRIXTRANSF_TPL_H_INCLUDED

